<?php
class Shipment {
    private $conn;
    private $table_name = "shipments";
    private $history_table = "shipment_status_history";

    public function __construct($db) {
        if (!$db) {
            throw new Exception('Database connection is required');
        }
        $this->conn = $db;
    }

    private function generateTrackingNumber() {
        $prefix = "TRK";
        $timestamp = date("YmdHis");
        $random = strtoupper(substr(uniqid(), -4));
        return $prefix . $timestamp . $random;
    }

    public function create($data) {
        try {
            if (!$this->conn) {
                throw new Exception('Database connection is not established');
            }

            $this->conn->beginTransaction();

            // Generate tracking number
            $tracking_number = $this->generateTrackingNumber();

            // Insert shipment
            $query = "INSERT INTO " . $this->table_name . "
                    (tracking_number, sender_name, sender_email, sender_phone,
                    receiver_name, receiver_email, receiver_phone,
                    origin, destination, payment_type, shipping_type,
                    pickup_datetime, estimated_delivery_datetime,
                    package_details, dimensions, weight_kg,
                    value_declared, service_type, current_status)
                    VALUES
                    (:tracking_number, :sender_name, :sender_email, :sender_phone,
                    :receiver_name, :receiver_email, :receiver_phone,
                    :origin, :destination, :payment_type, :shipping_type,
                    :pickup_datetime, :estimated_delivery_datetime,
                    :package_details, :dimensions, :weight_kg,
                    :value_declared, :service_type, 'Shipment Created')";

            $stmt = $this->conn->prepare($query);

            // Bind values
            $stmt->bindParam(":tracking_number", $tracking_number);
            $stmt->bindParam(":sender_name", $data->sender_name);
            $stmt->bindParam(":sender_email", $data->sender_email);
            $stmt->bindParam(":sender_phone", $data->sender_phone);
            $stmt->bindParam(":receiver_name", $data->receiver_name);
            $stmt->bindParam(":receiver_email", $data->receiver_email);
            $stmt->bindParam(":receiver_phone", $data->receiver_phone);
            $stmt->bindParam(":origin", $data->origin);
            $stmt->bindParam(":destination", $data->destination);
            $stmt->bindParam(":payment_type", $data->payment_type);
            $stmt->bindParam(":shipping_type", $data->shipping_type);
            $stmt->bindParam(":pickup_datetime", $data->pickup_datetime);
            $stmt->bindParam(":estimated_delivery_datetime", $data->estimated_delivery_datetime);
            $stmt->bindParam(":package_details", $data->package_details);
            $stmt->bindParam(":dimensions", $data->dimensions);
            $stmt->bindParam(":weight_kg", $data->weight_kg);
            $stmt->bindParam(":value_declared", $data->value_declared);
            $stmt->bindParam(":service_type", $data->service_type);

            if (!$stmt->execute()) {
                throw new Exception("Failed to insert shipment data");
            }

            $shipment_id = $this->conn->lastInsertId();

            // Add initial status history
            $history_query = "INSERT INTO " . $this->history_table . "
                            (shipment_id, status, remarks)
                            VALUES (:shipment_id, 'Shipment Created', 'Initial shipment creation')";
            
            $history_stmt = $this->conn->prepare($history_query);
            $history_stmt->bindParam(":shipment_id", $shipment_id);
            
            if (!$history_stmt->execute()) {
                throw new Exception("Failed to insert status history");
            }

            $this->conn->commit();

            return array(
                "id" => $shipment_id,
                "tracking_number" => $tracking_number
            );

        } catch (PDOException $e) {
            if ($this->conn) {
                $this->conn->rollBack();
            }
            error_log("Database error in Shipment::create: " . $e->getMessage());
            throw new Exception("Database error occurred while creating shipment");
        } catch (Exception $e) {
            if ($this->conn) {
                $this->conn->rollBack();
            }
            error_log("Error in Shipment::create: " . $e->getMessage());
            throw $e;
        }
    }

    public function list($page = 1, $per_page = 10) {
        try {
            $query = "SELECT * FROM " . $this->table_name . "
                    ORDER BY created_at DESC";
            
            $stmt = $this->conn->prepare($query);
            $stmt->execute();

            $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
            return $result ? $result : [];
        } catch (Exception $e) {
            error_log("Error in Shipment::list: " . $e->getMessage());
            throw new Exception("Failed to retrieve shipments");
        }
    }

    public function getById($id) {
        try {
            // Get shipment details
            $query = "SELECT * FROM " . $this->table_name . " WHERE id = ?";
            $stmt = $this->conn->prepare($query);
            $stmt->execute([$id]);
            $shipment = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$shipment) {
                return null;
            }

            // Get status history
            $history_query = "SELECT * FROM " . $this->history_table . "
                            WHERE shipment_id = ?
                            ORDER BY updated_at DESC";
            $history_stmt = $this->conn->prepare($history_query);
            $history_stmt->execute([$id]);
            $history = $history_stmt->fetchAll(PDO::FETCH_ASSOC);

            $shipment['status_history'] = $history;
            return $shipment;
        } catch (Exception $e) {
            error_log("Error in Shipment::getById: " . $e->getMessage());
            throw new Exception("Failed to retrieve shipment details");
        }
    }

    public function update($id, $data) {
        try {
            $updateFields = array();
            $values = array();

            foreach ($data as $key => $value) {
                if ($key !== 'id' && $key !== 'tracking_number') {
                    $updateFields[] = $key . " = ?";
                    $values[] = $value;
                }
            }

            if (empty($updateFields)) {
                return false;
            }

            $values[] = $id;
            $query = "UPDATE " . $this->table_name . "
                    SET " . implode(", ", $updateFields) . "
                    WHERE id = ?";

            $stmt = $this->conn->prepare($query);
            return $stmt->execute($values);
        } catch (Exception $e) {
            error_log("Error in Shipment::update: " . $e->getMessage());
            throw new Exception("Failed to update shipment");
        }
    }

    public function updateStatus($id, $new_status, $remarks = null) {
        try {
            $this->conn->beginTransaction();

            // Update current status in shipments table
            $query = "UPDATE " . $this->table_name . "
                    SET current_status = :status
                    WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":status", $new_status);
            $stmt->bindParam(":id", $id);
            
            if (!$stmt->execute()) {
                throw new Exception("Failed to update shipment status");
            }

            // Add new entry to status history
            $history_query = "INSERT INTO " . $this->history_table . "
                            (shipment_id, status, remarks)
                            VALUES (:shipment_id, :status, :remarks)";
            
            $history_stmt = $this->conn->prepare($history_query);
            $history_stmt->bindParam(":shipment_id", $id);
            $history_stmt->bindParam(":status", $new_status);
            $history_stmt->bindParam(":remarks", $remarks);
            
            if (!$history_stmt->execute()) {
                throw new Exception("Failed to add status history");
            }

            $this->conn->commit();
            return true;

        } catch (Exception $e) {
            if ($this->conn) {
                $this->conn->rollBack();
            }
            error_log("Error in Shipment::updateStatus: " . $e->getMessage());
            throw new Exception("Failed to update shipment status");
        }
    }

    public function getByTrackingNumber($tracking_number) {
        try {
            // Get shipment details
            $query = "SELECT 
                        id, tracking_number, sender_name, receiver_name,
                        origin, destination, shipping_type, service_type,
                        pickup_datetime, estimated_delivery_datetime,
                        package_details, dimensions, weight_kg,
                        current_status, created_at, updated_at
                    FROM " . $this->table_name . " 
                    WHERE tracking_number = ?";
                    
            $stmt = $this->conn->prepare($query);
            $stmt->execute([$tracking_number]);
            $shipment = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$shipment) {
                return null;
            }

            // Get status history
            $history_query = "SELECT status, remarks, updated_at
                            FROM " . $this->history_table . "
                            WHERE shipment_id = ?
                            ORDER BY updated_at DESC";
            $history_stmt = $this->conn->prepare($history_query);
            $history_stmt->execute([$shipment['id']]);
            $history = $history_stmt->fetchAll(PDO::FETCH_ASSOC);

            // Remove internal ID from response
            unset($shipment['id']);
            
            $shipment['status_history'] = $history;
            return $shipment;
        } catch (Exception $e) {
            error_log("Error in Shipment::getByTrackingNumber: " . $e->getMessage());
            throw new Exception("Failed to retrieve shipment details");
        }
    }
}
?> 