<?php
session_start();

// Check if system is already installed
if (file_exists('api/config/config.php') && file_exists('api/config/installed.txt')) {
    die('System is already installed. For security reasons, please remove the install.php file.');
}

// Get any stored messages
$error = isset($_SESSION['install_error']) ? $_SESSION['install_error'] : null;
$success_message = isset($_SESSION['install_message']) ? $_SESSION['install_message'] : null;

// Clear session messages
unset($_SESSION['install_error']);
unset($_SESSION['install_message']);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $shouldCleanup = true; // Flag to determine if cleanup is needed
    
    try {
        // Validate required fields
        $required_fields = [
            'db_host' => 'Database Host',
            'db_name' => 'Database Name',
            'db_user' => 'Database User',
            'db_pass' => 'Database Password',
            'smtp_host' => 'SMTP Host',
            'smtp_port' => 'SMTP Port',
            'smtp_user' => 'SMTP Username',
            'smtp_pass' => 'SMTP Password',
            'smtp_from_email' => 'From Email',
            'smtp_from_name' => 'From Name',
            'admin_username' => 'Admin Username',
            'admin_password' => 'Admin Password',
            'admin_email' => 'Admin Email',
            'admin_full_name' => 'Admin Full Name'
        ];

        foreach ($required_fields as $field => $label) {
            if (empty($_POST[$field])) {
                throw new Exception("$label is required");
            }
        }

        // Validate email formats
        if (!filter_var($_POST['smtp_from_email'], FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Invalid SMTP From Email format');
        }
        if (!filter_var($_POST['admin_email'], FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Invalid Admin Email format');
        }

        // Validate password length
        if (strlen($_POST['admin_password']) < 8) {
            throw new Exception('Admin password must be at least 8 characters long');
        }

        // Database configuration
        $dbConfig = [
            'host' => $_POST['db_host'],
            'name' => $_POST['db_name'],
            'user' => $_POST['db_user'],
            'pass' => $_POST['db_pass']
        ];

        // Test database connection
        try {
            $conn = new PDO(
                "mysql:host={$dbConfig['host']};charset=utf8mb4",
                $dbConfig['user'],
                $dbConfig['pass']
            );
            $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch (PDOException $e) {
            throw new Exception('Database connection failed: ' . $e->getMessage());
        }

        // Create database if not exists
        try {
            $conn->exec("CREATE DATABASE IF NOT EXISTS `{$dbConfig['name']}`");
            $conn->exec("USE `{$dbConfig['name']}`");
        } catch (PDOException $e) {
            throw new Exception('Failed to create database: ' . $e->getMessage());
        }

        // Check if tables already exist and have correct structure
        $tablesExist = false;
        $needsUpdate = false;
        try {
            $stmt = $conn->query("SHOW TABLES LIKE 'admins'");
            $tablesExist = $stmt->rowCount() > 0;
            
            if ($tablesExist) {
                // Check if admins table has the correct structure
                $stmt = $conn->query("DESCRIBE admins");
                $existingColumns = $stmt->fetchAll(PDO::FETCH_COLUMN);
                
                // Define required columns with their definitions
                $requiredColumns = [
                    'id' => 'INT PRIMARY KEY AUTO_INCREMENT',
                    'username' => 'VARCHAR(50) UNIQUE NOT NULL',
                    'password' => 'VARCHAR(255) NOT NULL',
                    'email' => 'VARCHAR(100) UNIQUE NOT NULL',
                    'full_name' => 'VARCHAR(100) NOT NULL',
                    'created_at' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP'
                ];
                
                // Add any missing columns
                foreach ($requiredColumns as $column => $definition) {
                    if (!in_array($column, $existingColumns)) {
                        try {
                            $conn->exec("ALTER TABLE admins ADD COLUMN $column $definition");
                            $needsUpdate = true;
                        } catch (PDOException $alterError) {
                            throw new Exception("Failed to add column $column: " . $alterError->getMessage());
                        }
                    }
                }
            }
        } catch (PDOException $e) {
            // Only throw if it's not the initial table check
            if ($tablesExist) {
                throw new Exception('Failed to check or update table structure: ' . $e->getMessage());
            }
        }

        // Execute the schema if tables don't exist
        if (!$tablesExist) {
            try {
                if (!file_exists('schema.sql')) {
                    throw new Exception('Schema file (schema.sql) not found');
                }
                $schema = file_get_contents('schema.sql');
                $conn->exec($schema);
            } catch (PDOException $e) {
                // Only throw if it's not a "table already exists" error
                if (strpos($e->getMessage(), '1050') === false) {
                    throw new Exception('Failed to create database tables: ' . $e->getMessage());
                }
            }
        }

        // If we added new columns, we need to set default values for existing rows
        if ($needsUpdate) {
            try {
                // Update existing rows with default values if email and full_name were added
                $stmt = $conn->prepare("
                    UPDATE admins 
                    SET email = CONCAT(username, '@temp.com'),
                        full_name = username
                    WHERE email IS NULL OR full_name IS NULL
                ");
                $stmt->execute();
            } catch (PDOException $e) {
                throw new Exception('Failed to update existing admin records: ' . $e->getMessage());
            }
        }

        // Check if admin already exists
        try {
            $stmt = $conn->prepare("SELECT COUNT(*) FROM admins WHERE username = ? OR email = ?");
            $stmt->execute([$_POST['admin_username'], $_POST['admin_email']]);
            if ($stmt->fetchColumn() > 0) {
                throw new Exception('An admin with this username or email already exists');
            }
        } catch (PDOException $e) {
            throw new Exception('Failed to check for existing admin: ' . $e->getMessage());
        }

        // Create admin account
        try {
            $hashedPassword = password_hash($_POST['admin_password'], PASSWORD_DEFAULT);
            $stmt = $conn->prepare("
                INSERT INTO admins (username, password, email, full_name)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([
                $_POST['admin_username'],
                $hashedPassword,
                $_POST['admin_email'],
                $_POST['admin_full_name']
            ]);
        } catch (PDOException $e) {
            throw new Exception('Failed to create admin account: ' . $e->getMessage());
        }

        // Create config.php file
        $configContent = "<?php
return [
    'db' => [
        'host' => '{$dbConfig['host']}',
        'name' => '{$dbConfig['name']}',
        'user' => '{$dbConfig['user']}',
        'pass' => '{$dbConfig['pass']}'
    ],
    'smtp' => [
        'host' => '{$_POST['smtp_host']}',
        'port' => '{$_POST['smtp_port']}',
        'user' => '{$_POST['smtp_user']}',
        'pass' => '{$_POST['smtp_pass']}',
        'from_email' => '{$_POST['smtp_from_email']}',
        'from_name' => '{$_POST['smtp_from_name']}'
    ]
];";

        try {
            if (!is_dir('api/config')) {
                if (!mkdir('api/config', 0755, true)) {
                    throw new Exception('Failed to create config directory');
                }
            }
            if (!file_put_contents('api/config/config.php', $configContent)) {
                throw new Exception('Failed to write config file');
            }
        } catch (Exception $e) {
            throw new Exception('Configuration error: ' . $e->getMessage());
        }

        // Mark as installed
        try {
            if (!file_put_contents('api/config/installed.txt', date('Y-m-d H:i:s'))) {
                throw new Exception('Failed to create installation marker');
            }
        } catch (Exception $e) {
            throw new Exception('Failed to complete installation: ' . $e->getMessage());
        }

        // Installation successful
        $_SESSION['install_success'] = true;
        $_SESSION['install_message'] = 'Installation completed successfully!';
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;

    } catch (Exception $e) {
        $error = $e->getMessage();
        
        // Only cleanup config files, never drop the database
        try {
            if (file_exists('api/config/config.php')) {
                unlink('api/config/config.php');
            }
            if (file_exists('api/config/installed.txt')) {
                unlink('api/config/installed.txt');
            }
        } catch (Exception $cleanupError) {
            $error .= "\nCleanup failed: " . $cleanupError->getMessage();
        }
        
        // Store error in session to display after redirect
        $_SESSION['install_error'] = $error;
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Courier Tracking System Installation</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#2563eb',
                        secondary: '#475569'
                    }
                }
            }
        }
    </script>
    <style>
        input {
            background-color: white !important;
        }
        .form-input {
            display: block;
            width: 100%;
            padding: 0.5rem 0.75rem;
            border: 1px solid #d1d5db;
            border-radius: 0.375rem;
            box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
        }
        .form-input:focus {
            outline: none;
            border-color: #2563eb;
            ring: 1px #2563eb;
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Toast Container -->
    <div id="toastContainer" class="fixed bottom-4 right-4 z-50 space-y-2"></div>

    <div class="min-h-screen flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
        <div class="max-w-3xl w-full space-y-8">
            <?php if (isset($_SESSION['install_success'])): ?>
                <div class="bg-green-50 border border-green-200 rounded-lg p-6 text-center">
                    <h2 class="text-2xl font-bold text-green-800 mb-4">Installation Successful!</h2>
                    <p class="text-green-700 mb-4">The system has been installed successfully.</p>
                    <div class="space-y-2">
                        <p class="text-green-700">Please:</p>
                        <ol class="list-decimal list-inside text-green-700">
                            <li>Delete the install.php file</li>
                            <li>Proceed to <a href="admin.html" class="text-green-800 underline">admin login</a></li>
                        </ol>
                    </div>
                </div>
            <?php else: ?>
                <div class="bg-white rounded-lg shadow-sm p-6">
                    <div class="text-center mb-8">
                        <h2 class="text-3xl font-bold text-gray-900">System Installation</h2>
                        <p class="mt-2 text-gray-600">Configure your courier tracking system</p>
                    </div>

                    <form method="POST" id="installForm" class="space-y-8">
                        <!-- Database Configuration -->
                        <div>
                            <h3 class="text-lg font-medium text-gray-900 mb-4">Database Configuration</h3>
                            <div class="grid grid-cols-1 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Database Host</label>
                                    <input type="text" name="db_host" required value="localhost" class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Database Name</label>
                                    <input type="text" name="db_name" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Database User</label>
                                    <input type="text" name="db_user" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Database Password</label>
                                    <input type="password" name="db_pass" required class="form-input bg-white">
                                </div>
                            </div>
                        </div>

                        <!-- SMTP Configuration -->
                        <div>
                            <h3 class="text-lg font-medium text-gray-900 mb-4">SMTP Configuration</h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">SMTP Host</label>
                                    <input type="text" name="smtp_host" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">SMTP Port</label>
                                    <input type="number" name="smtp_port" required value="587" class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">SMTP Username</label>
                                    <input type="text" name="smtp_user" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">SMTP Password</label>
                                    <input type="password" name="smtp_pass" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">From Email</label>
                                    <input type="email" name="smtp_from_email" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">From Name</label>
                                    <input type="text" name="smtp_from_name" required class="form-input bg-white">
                                </div>
                            </div>
                        </div>

                        <!-- Admin Account -->
                        <div>
                            <h3 class="text-lg font-medium text-gray-900 mb-4">Admin Account</h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Username</label>
                                    <input type="text" name="admin_username" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Password</label>
                                    <input type="password" name="admin_password" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Email</label>
                                    <input type="email" name="admin_email" required class="form-input bg-white">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Full Name</label>
                                    <input type="text" name="admin_full_name" required class="form-input bg-white">
                                </div>
                            </div>
                        </div>

                        <div class="pt-4">
                            <button type="submit"
                                    class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                Install System
                            </button>
                        </div>
                    </form>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Toast notification system
        function showToast(message, type = 'success', duration = 3000) {
            const container = document.getElementById('toastContainer');
            const toast = document.createElement('div');
            const bgColor = type === 'success' ? 'bg-green-50' : 'bg-red-50';
            const textColor = type === 'success' ? 'text-green-800' : 'text-red-800';
            const borderColor = type === 'success' ? 'border-green-200' : 'border-red-200';
            
            // Split message into lines if it contains newlines
            const messages = message.split('\n').filter(msg => msg.trim());
            
            toast.className = `flex items-start p-4 mb-4 border rounded-lg shadow-sm ${bgColor} ${textColor} ${borderColor}`;
            toast.innerHTML = `
                <div class="flex-shrink-0 mt-0.5">
                    ${type === 'success' ? 
                        '<svg class="h-5 w-5 text-green-400" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/></svg>' :
                        '<svg class="h-5 w-5 text-red-400" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/></svg>'
                    }
                </div>
                <div class="ml-3">
                    ${messages.map(msg => `<div class="font-medium">${msg}</div>`).join('')}
                </div>
                <button class="ml-auto flex-shrink-0 -mr-1 -mt-1" onclick="this.parentElement.remove()">
                    <svg class="h-5 w-5 text-gray-400 hover:text-gray-500" fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"/>
                    </svg>
                </button>
            `;

            container.appendChild(toast);

            if (duration > 0) {
                setTimeout(() => {
                    if (toast.parentElement) {
                        toast.remove();
                    }
                }, duration);
            }
        }

        // Form validation
        document.getElementById('installForm').addEventListener('submit', function(e) {
            const validateField = (name, label, type = 'text') => {
                const field = this.querySelector(`input[name="${name}"]`);
                const value = field.value.trim();
                
                if (!value) {
                    showToast(`${label} is required`, 'error', 5000);
                    field.focus();
                    return false;
                }
                
                if (type === 'email' && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) {
                    showToast(`${label} must be a valid email address`, 'error', 5000);
                    field.focus();
                    return false;
                }
                
                return true;
            };

            // Prevent default form submission
            e.preventDefault();
            
            // Validate all required fields
            const fields = [
                ['db_host', 'Database Host'],
                ['db_name', 'Database Name'],
                ['db_user', 'Database Username'],
                ['db_pass', 'Database Password'],
                ['smtp_host', 'SMTP Host'],
                ['smtp_port', 'SMTP Port'],
                ['smtp_user', 'SMTP Username'],
                ['smtp_pass', 'SMTP Password'],
                ['smtp_from_email', 'From Email', 'email'],
                ['smtp_from_name', 'From Name'],
                ['admin_username', 'Admin Username'],
                ['admin_password', 'Admin Password'],
                ['admin_email', 'Admin Email', 'email'],
                ['admin_full_name', 'Admin Full Name']
            ];

            for (const [name, label, type] of fields) {
                if (!validateField(name, label, type)) {
                    return;
                }
            }

            // Validate password length
            const password = this.querySelector('input[name="admin_password"]').value;
            if (password.length < 8) {
                showToast('Admin password must be at least 8 characters long', 'error', 5000);
                this.querySelector('input[name="admin_password"]').focus();
                return;
            }

            // Submit the form if all validations pass
            this.submit();
        });

        // Show PHP errors and success messages from session
        <?php if ($error): ?>
            showToast(`<?php echo str_replace("'", "\\'", str_replace("\n", "\\n", $error)); ?>`, 'error', 0);
        <?php endif; ?>
        
        <?php if ($success_message): ?>
            showToast(`<?php echo str_replace("'", "\\'", $success_message); ?>`, 'success', 5000);
        <?php endif; ?>
    </script>
</body>
</html> 