<?php
require_once __DIR__ . '/../vendor/autoload.php';
use Picqer\Barcode\BarcodeGeneratorPNG;

class BarcodeGenerator {
    private $generator;

    public function __construct() {
        $this->generator = new BarcodeGeneratorPNG();
    }

    /**
     * Generate a Code 128 barcode for the given tracking number
     * 
     * @param string $tracking_number The tracking number to encode
     * @param int $height Height of the barcode in pixels
     * @param int $width_factor Width multiplier for bars
     * @return string Binary image data
     */
    public function generateBarcode($tracking_number, $height = 100, $width_factor = 2) {
        try {
            return $this->generator->getBarcode(
                $tracking_number,
                $this->generator::TYPE_CODE_128,
                $width_factor,
                $height
            );
        } catch (Exception $e) {
            throw new Exception("Failed to generate barcode: " . $e->getMessage());
        }
    }

    /**
     * Output barcode image directly to browser with proper headers
     * 
     * @param string $tracking_number The tracking number to encode
     * @param int $height Height of the barcode in pixels
     * @param int $width_factor Width multiplier for bars
     */
    public function outputBarcode($tracking_number, $height = 100, $width_factor = 2) {
        try {
            // Set headers for image output
            header('Content-Type: image/png');
            header('Cache-Control: public, max-age=86400'); // Cache for 24 hours
            header('Content-Disposition: inline; filename="' . $tracking_number . '.png"');
            
            // Output the barcode image
            echo $this->generateBarcode($tracking_number, $height, $width_factor);
        } catch (Exception $e) {
            // If there's an error, clear any output and return error image
            ob_clean();
            $this->outputErrorImage("Failed to generate barcode");
        }
    }

    /**
     * Output a simple error image
     * 
     * @param string $message Error message to display
     */
    private function outputErrorImage($message) {
        $img = imagecreate(400, 50);
        $bg = imagecolorallocate($img, 255, 255, 255);
        $textcolor = imagecolorallocate($img, 255, 0, 0);
        
        imagestring($img, 5, 10, 20, $message, $textcolor);
        
        header('Content-Type: image/png');
        imagepng($img);
        imagedestroy($img);
    }
}
?> 