<?php
require_once __DIR__ . '/../../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class MailHelper {
    private $mailer;
    private $config;
    private $db;
    private $isEnabled;

    public function __construct($db) {
        $this->db = $db;
        $this->config = require __DIR__ . '/../config/mail.php';
        $this->isEnabled = !empty($this->config['username']) && !empty($this->config['password']);
        
        if ($this->isEnabled) {
            $this->initializeMailer();
        }
    }

    private function initializeMailer() {
        try {
            $this->mailer = new PHPMailer(true);

            // Server settings
            $this->mailer->isSMTP();
            $this->mailer->Host = $this->config['host'];
            $this->mailer->SMTPAuth = true;
            $this->mailer->Username = $this->config['username'];
            $this->mailer->Password = $this->config['password'];
            $this->mailer->SMTPSecure = $this->config['encryption'];
            $this->mailer->Port = $this->config['port'];

            // Default sender
            $this->mailer->setFrom($this->config['from_email'], $this->config['from_name']);
        } catch (Exception $e) {
            error_log("Failed to initialize mailer: " . $e->getMessage());
            $this->isEnabled = false;
        }
    }

    private function getEmailTemplate($key) {
        try {
            $stmt = $this->db->prepare("SELECT setting_value FROM app_settings WHERE setting_key = ?");
            $stmt->execute([$key]);
            return $stmt->fetchColumn();
        } catch (Exception $e) {
            error_log("Failed to get email template: " . $e->getMessage());
            return false;
        }
    }

    private function replacePlaceholders($template, $data) {
        foreach ($data as $key => $value) {
            $template = str_replace('{{' . $key . '}}', $value, $template);
        }
        return $template;
    }

    public function sendNewShipmentEmail($shipmentData) {
        if (!$this->isEnabled) {
            error_log("Email sending is disabled - SMTP credentials not configured");
            return false;
        }

        try {
            $template = $this->getEmailTemplate('email_template_new_shipment');
            if (!$template) {
                error_log("New shipment email template not found");
                return false;
            }

            // Prepare data for template
            $templateData = [
                'tracking_number' => $shipmentData['tracking_number'],
                'receiver_name' => $shipmentData['receiver_name'],
                'sender_name' => $shipmentData['sender_name']
            ];

            // Replace placeholders in template
            $emailContent = $this->replacePlaceholders($template, $templateData);

            // Reset recipients
            $this->mailer->clearAddresses();

            // Add recipients
            if (!empty($shipmentData['receiver_email'])) {
                $this->mailer->addAddress($shipmentData['receiver_email'], $shipmentData['receiver_name']);
            }
            if (!empty($shipmentData['sender_email'])) {
                $this->mailer->addCC($shipmentData['sender_email'], $shipmentData['sender_name']);
            }

            // Content
            $this->mailer->isHTML(true);
            $this->mailer->Subject = "New Shipment Created - {$shipmentData['tracking_number']}";
            $this->mailer->Body = $emailContent;
            $this->mailer->AltBody = strip_tags($emailContent);

            return $this->mailer->send();
        } catch (Exception $e) {
            error_log("Failed to send new shipment email: " . $e->getMessage());
            return false;
        }
    }

    public function sendStatusUpdateEmail($shipmentData, $newStatus) {
        if (!$this->isEnabled) {
            error_log("Email sending is disabled - SMTP credentials not configured");
            return false;
        }

        try {
            $template = $this->getEmailTemplate('email_template_status_update');
            if (!$template) {
                error_log("Status update email template not found");
                return false;
            }

            // Prepare data for template
            $templateData = [
                'tracking_number' => $shipmentData['tracking_number'],
                'receiver_name' => $shipmentData['receiver_name'],
                'new_status' => $newStatus
            ];

            // Replace placeholders in template
            $emailContent = $this->replacePlaceholders($template, $templateData);

            // Reset recipients
            $this->mailer->clearAddresses();

            // Add recipient
            if (!empty($shipmentData['receiver_email'])) {
                $this->mailer->addAddress($shipmentData['receiver_email'], $shipmentData['receiver_name']);
            }

            // Content
            $this->mailer->isHTML(true);
            $this->mailer->Subject = "Shipment Status Update - {$shipmentData['tracking_number']}";
            $this->mailer->Body = $emailContent;
            $this->mailer->AltBody = strip_tags($emailContent);

            return $this->mailer->send();
        } catch (Exception $e) {
            error_log("Failed to send status update email: " . $e->getMessage());
            return false;
        }
    }
} 