<?php
// Required headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database, auth middleware and shipment model
include_once '../../../../config/db.php';
include_once '../../../../includes/auth_middleware.php';
include_once '../../../../models/Shipment.php';
require_once '../../../auth/check_admin_auth.php';
require_once '../../../helpers/MailHelper.php';

// Check authentication
AuthMiddleware::isAuthenticated();

// Check if user is authenticated
if (!isAdminAuthenticated()) {
    http_response_code(401);
    echo json_encode([
        'status' => 'error',
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Get database connection
$database = new Database();
$db = $database->getConnection();

// Get posted data
$data = json_decode(file_get_contents("php://input"));

// Validate required fields
if (!isset($data->shipment_id) || empty($data->shipment_id) || 
    !isset($data->status) || empty($data->status)) {
    http_response_code(400);
    echo json_encode(array(
        "status" => "error",
        "message" => "Shipment ID and status are required"
    ));
    exit();
}

try {
    $shipment = new Shipment($db);
    
    // First check if shipment exists
    $existing = $shipment->getById($data->shipment_id);
    if (!$existing) {
        http_response_code(404);
        echo json_encode(array(
            "status" => "error",
            "message" => "Shipment not found"
        ));
        exit();
    }

    // Start transaction
    $db->getConnection()->beginTransaction();

    // Update status
    $remarks = isset($data->remarks) ? $data->remarks : null;
    if ($shipment->updateStatus($data->shipment_id, $data->status, $remarks)) {
        // Add status to history
        $historyQuery = "INSERT INTO shipment_status_history (
                            shipment_id, status, location, remarks
                        ) VALUES (
                            :shipment_id, :status, :location, :remarks
                        )";
        
        $historyStmt = $db->getConnection()->prepare($historyQuery);
        $historyStmt->execute([
            ':shipment_id' => $data->shipment_id,
            ':status' => $data->status,
            ':location' => $data->location ?? null,
            ':remarks' => $remarks ?? 'Status updated'
        ]);

        // Commit transaction
        $db->getConnection()->commit();

        // Send email notification
        $mailHelper = new MailHelper($db->getConnection());
        $emailSent = $mailHelper->sendStatusUpdateEmail([
            'tracking_number' => $existing['tracking_number'],
            'receiver_name' => $existing['receiver_name'],
            'receiver_email' => $existing['receiver_email']
        ], $data->status);

        http_response_code(200);
        echo json_encode(array(
            "status" => "success",
            "message" => "Shipment status updated successfully",
            "email_sent" => $emailSent
        ));
    } else {
        http_response_code(500);
        echo json_encode(array(
            "status" => "error",
            "message" => "Unable to update shipment status"
        ));
    }

} catch (PDOException $e) {
    if ($db->getConnection()) {
        $db->getConnection()->rollBack();
    }
    http_response_code(500);
    echo json_encode(array(
        "status" => "error",
        "message" => "Database error: " . $e->getMessage()
    ));
} catch (Exception $e) {
    if ($db->getConnection()) {
        $db->getConnection()->rollBack();
    }
    http_response_code(400);
    echo json_encode(array(
        "status" => "error",
        "message" => $e->getMessage()
    ));
}
?> 