<?php
// Required headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error.log');

// Include database and auth check
require_once '../../config/database.php';
require_once '../../auth/check_admin_auth.php';
require_once '../../models/Shipment.php';
require_once '../../helpers/MailHelper.php';

// Check if user is authenticated
if (!isAdminAuthenticated()) {
    error_log("Unauthorized access attempt in create shipment");
    http_response_code(401);
    echo json_encode([
        'status' => 'error',
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Get posted data
$raw_data = file_get_contents("php://input");
error_log("Received data: " . $raw_data);

$data = json_decode($raw_data);

// Check if data is valid JSON
if (json_last_error() !== JSON_ERROR_NONE) {
    error_log("JSON decode error: " . json_last_error_msg());
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => 'Invalid JSON data: ' . json_last_error_msg()
    ]);
    exit;
}

// Validate required fields
$required_fields = [
    'sender_name', 'sender_email', 'sender_phone',
    'receiver_name', 'receiver_email', 'receiver_phone',
    'origin', 'destination', 'payment_type', 'shipping_type',
    'pickup_datetime', 'estimated_delivery_datetime',
    'package_details', 'dimensions', 'weight_kg',
    'value_declared', 'service_type'
];

$errors = array();
foreach ($required_fields as $field) {
    if (!isset($data->$field) || empty($data->$field)) {
        $errors[] = "$field is required";
    }
}

if (!empty($errors)) {
    error_log("Validation errors: " . implode(", ", $errors));
    http_response_code(400);
    echo json_encode([
        "status" => "error",
        "message" => "Validation failed",
        "errors" => $errors
    ]);
    exit;
}

try {
    // Get database connection
    $database = new Database();
    $db = $database->getConnection();

    // Create shipment instance
    $shipment = new Shipment($db);
    
    // Create the shipment
    $result = $shipment->create($data);

    if ($result) {
        try {
            // Send email notifications
            $mailHelper = new MailHelper($db);
            $emailSent = $mailHelper->sendNewShipmentEmail([
                'tracking_number' => $result['tracking_number'],
                'sender_name' => $data->sender_name,
                'sender_email' => $data->sender_email,
                'receiver_name' => $data->receiver_name,
                'receiver_email' => $data->receiver_email
            ]);
        } catch (Exception $e) {
            error_log("Email sending failed: " . $e->getMessage());
            $emailSent = false;
        }

        http_response_code(201);
        echo json_encode([
            "status" => "success",
            "message" => "Shipment created successfully",
            "data" => $result,
            "email_sent" => $emailSent
        ]);
    } else {
        throw new Exception("Failed to create shipment");
    }
} catch (PDOException $e) {
    error_log("Database error creating shipment: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => "Database error occurred while creating shipment",
        "error" => "Please check your database connection and try again"
    ]);
} catch (Exception $e) {
    error_log("Error creating shipment: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => "Unable to create shipment",
        "error" => $e->getMessage()
    ]);
}
?> 