<?php
require_once '../../config/database.php';
require_once '../../auth/check_admin_auth.php';

header('Content-Type: application/json');

// Check if user is authenticated
if (!isAdminAuthenticated()) {
    http_response_code(401);
    echo json_encode([
        'status' => 'error',
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!is_array($input)) {
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => 'Invalid input format. Expected array of settings.'
    ]);
    exit;
}

try {
    $db = new Database();
    $conn = $db->getConnection();

    // Start transaction
    $conn->beginTransaction();

    // Prepare the upsert query (insert or update)
    $query = "INSERT INTO app_settings (setting_key, setting_value) 
              VALUES (:key, :value) 
              ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)";
    $stmt = $conn->prepare($query);

    // Process each setting
    foreach ($input as $setting) {
        if (!isset($setting['key']) || !isset($setting['value'])) {
            throw new Exception('Invalid setting format. Each setting must have "key" and "value".');
        }

        $stmt->execute([
            ':key' => $setting['key'],
            ':value' => $setting['value']
        ]);
    }

    // Commit transaction
    $conn->commit();

    echo json_encode([
        'status' => 'success',
        'message' => 'Settings updated successfully'
    ]);

} catch (PDOException $e) {
    if ($conn) {
        $conn->rollBack();
    }
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} catch (Exception $e) {
    if ($conn) {
        $conn->rollBack();
    }
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
} 