<?php
require_once '../../config/database.php';
require_once '../../auth/check_admin_auth.php';

header('Content-Type: application/json');

// Check if user is authenticated
if (!isAdminAuthenticated()) {
    http_response_code(401);
    echo json_encode([
        'status' => 'error',
        'message' => 'Unauthorized access'
    ]);
    exit;
}

try {
    $db = new Database();
    $conn = $db->getConnection();

    // Initialize response array
    $analytics = [
        'total_shipments' => 0,
        'shipments_by_status' => [],
        'recent_shipments' => 0
    ];

    // 1. Get total number of shipments
    $totalQuery = "SELECT COUNT(*) as total FROM shipments";
    $stmt = $conn->query($totalQuery);
    $analytics['total_shipments'] = (int)$stmt->fetchColumn();

    // 2. Get number of shipments by status
    $statusQuery = "SELECT status, COUNT(*) as count 
                   FROM shipments 
                   GROUP BY status";
    $stmt = $conn->query($statusQuery);
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $analytics['shipments_by_status'][$row['status']] = (int)$row['count'];
    }

    // 3. Get number of shipments in last 7 days
    $recentQuery = "SELECT COUNT(*) as recent 
                   FROM shipments 
                   WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)";
    $stmt = $conn->query($recentQuery);
    $analytics['recent_shipments'] = (int)$stmt->fetchColumn();

    // 4. Additional useful metrics
    
    // Get today's shipments
    $todayQuery = "SELECT COUNT(*) as today 
                  FROM shipments 
                  WHERE DATE(created_at) = CURDATE()";
    $stmt = $conn->query($todayQuery);
    $analytics['today_shipments'] = (int)$stmt->fetchColumn();

    // Get pending deliveries (not in final states)
    $pendingQuery = "SELECT COUNT(*) as pending 
                    FROM shipments 
                    WHERE status NOT IN ('Delivered', 'Returned', 'Cancelled')";
    $stmt = $conn->query($pendingQuery);
    $analytics['pending_deliveries'] = (int)$stmt->fetchColumn();

    // Get daily shipments for the last 7 days
    $dailyTrendQuery = "SELECT 
                           DATE(created_at) as date,
                           COUNT(*) as count
                       FROM shipments
                       WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
                       GROUP BY DATE(created_at)
                       ORDER BY date DESC";
    $stmt = $conn->query($dailyTrendQuery);
    $analytics['daily_trend'] = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $analytics['daily_trend'][$row['date']] = (int)$row['count'];
    }

    echo json_encode([
        'status' => 'success',
        'data' => $analytics
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Server error: ' . $e->getMessage()
    ]);
} 